<?php
/**
 * Debug Payroll Module
 * Access this file via: your-site.com/wp-content/plugins/hr-management/debug_payroll.php
 * Or run via WP Admin → Tools → WP Debugging or similar
 */

// Load WordPress
require_once dirname(__FILE__) . '/../../../../wp-load.php';

if (!current_user_can('manage_options')) {
    wp_die('Unauthorized. Please log in as admin.');
}

global $wpdb;

echo "<h1>HRM Payroll Debug Report</h1>";
echo "<style>
    body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; padding: 20px; max-width: 1200px; margin: 0 auto; }
    .section { background: #f9f9f9; border: 1px solid #ddd; padding: 20px; margin: 20px 0; border-radius: 8px; }
    .success { color: #0a0; font-weight: bold; }
    .error { color: #d00; font-weight: bold; }
    .warning { color: #f60; font-weight: bold; }
    table { border-collapse: collapse; width: 100%; margin: 10px 0; }
    th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
    th { background: #eee; }
    pre { background: #f5f5f5; padding: 10px; overflow-x: auto; }
</style>";

// 1. Check if payroll class exists
echo "<div class='section'>";
echo "<h2>1. Payroll Class Status</h2>";
if (class_exists('HRM_Payroll')) {
    echo "<p class='success'>✓ HRM_Payroll class exists</p>";
    
    // Check if init method exists
    if (method_exists('HRM_Payroll', 'init')) {
        echo "<p class='success'>✓ HRM_Payroll::init() method exists</p>";
    } else {
        echo "<p class='error'>✗ HRM_Payroll::init() method NOT FOUND</p>";
    }
} else {
    echo "<p class='error'>✗ HRM_Payroll class NOT FOUND</p>";
}
echo "</div>";

// 2. Check database tables
echo "<div class='section'>";
echo "<h2>2. Database Tables</h2>";

$tables_to_check = [
    $wpdb->prefix . 'hrm_payroll',
    $wpdb->prefix . 'hrm_salary_structures',
    $wpdb->prefix . 'hrm_attendance'
];

foreach ($tables_to_check as $table) {
    $exists = $wpdb->get_var("SHOW TABLES LIKE '$table'");
    if ($exists) {
        $count = $wpdb->get_var("SELECT COUNT(*) FROM $table");
        echo "<p class='success'>✓ Table <code>$table</code> exists ($count records)</p>";
    } else {
        echo "<p class='error'>✗ Table <code>$table</code> NOT FOUND</p>";
    }
}
echo "</div>";

// 3. Check employees
echo "<div class='section'>";
echo "<h2>3. Employees Status</h2>";

$employees = $wpdb->get_results("SELECT ID, post_title FROM {$wpdb->posts} WHERE post_type = 'hrm_employee' AND post_status = 'publish' LIMIT 10");

if (empty($employees)) {
    echo "<p class='warning'>⚠ No employees found. You need to add employees first before generating payroll.</p>";
} else {
    echo "<p class='success'>✓ Found " . count($employees) . " employee(s)</p>";
    echo "<table>";
    echo "<tr><th>ID</th><th>Name</th><th>Has Salary Structure</th></tr>";
    
    foreach ($employees as $emp) {
        $has_structure = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$wpdb->prefix}hrm_salary_structures WHERE employee_id = %d",
            $emp->ID
        ));
        $structure_status = $has_structure 
            ? "<span class='success'>Yes</span>" 
            : "<span class='warning'>No - Needs Setup!</span>";
        
        echo "<tr>";
        echo "<td>{$emp->ID}</td>";
        echo "<td>" . esc_html($emp->post_title) . "</td>";
        echo "<td>{$structure_status}</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    if (!$has_structure) {
        echo "<p class='warning'>⚠ <strong>Important:</strong> Employees need salary structures before payroll can be generated. Go to Payroll → Salary Structures and set up each employee's salary.</p>";
    }
}
echo "</div>";

// 4. Check payroll settings
echo "<div class='section'>";
echo "<h2>4. Payroll Settings</h2>";

$settings = get_option('hrm_payroll_settings', []);

if (empty($settings)) {
    echo "<p class='warning'>⚠ No payroll settings found. Using defaults.</p>";
    $settings = [
        'working_days' => 22,
        'currency' => '$',
        'tax_slabs' => []
    ];
} else {
    echo "<p class='success'>✓ Payroll settings found</p>";
}

echo "<pre>" . print_r($settings, true) . "</pre>";
echo "</div>";

// 5. Check recent payroll records
echo "<div class='section'>";
echo "<h2>5. Recent Payroll Records</h2>";

$payroll_table = $wpdb->prefix . 'hrm_payroll';
$table_exists = $wpdb->get_var("SHOW TABLES LIKE '$payroll_table'");

if ($table_exists) {
    $recent = $wpdb->get_results("SELECT pr.*, p.post_title as employee_name 
        FROM {$wpdb->prefix}hrm_payroll pr 
        LEFT JOIN {$wpdb->posts} p ON pr.employee_id = p.ID 
        ORDER BY pr.created_at DESC LIMIT 5");
    
    if (empty($recent)) {
        echo "<p class='warning'>⚠ No payroll records yet. Generate payroll for employees.</p>";
    } else {
        echo "<table>";
        echo "<tr><th>ID</th><th>Employee</th><th>Month</th><th>Net Salary</th><th>Status</th><th>Created</th></tr>";
        foreach ($recent as $row) {
            echo "<tr>";
            echo "<td>{$row->id}</td>";
            echo "<td>" . esc_html($row->employee_name) . "</td>";
            echo "<td>{$row->payroll_month}</td>";
            echo "<td>\${$row->net_salary}</td>";
            echo "<td>{$row->status}</td>";
            echo "<td>{$row->created_at}</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
}
echo "</div>";

// 6. Check AJAX handlers
echo "<div class='section'>";
echo "<h2>6. AJAX Handlers Registration</h2>";

$expected_actions = [
    'wp_ajax_hrm_save_salary_structure',
    'wp_ajax_hrm_generate_payroll',
    'wp_ajax_hrm_bulk_generate_payroll',
    'wp_ajax_hrm_update_payroll_status',
    'wp_ajax_hrm_get_payslip',
    'wp_ajax_hrm_delete_payroll',
    'wp_ajax_hrm_save_payroll_settings'
];

global $wp_filter;

foreach ($expected_actions as $action) {
    if (isset($wp_filter[$action]) && !empty($wp_filter[$action]->callbacks)) {
        echo "<p class='success'>✓ $action is registered</p>";
    } else {
        echo "<p class='error'>✗ $action NOT registered!</p>";
    }
}
echo "</div>";

// 7. Check for PHP errors
echo "<div class='section'>";
echo "<h2>7. Last PHP Errors (if any)</h2>";

$error_log = ini_get('error_log');
if ($error_log && file_exists($error_log) && is_readable($error_log)) {
    $lines = file($error_log);
    $last_lines = array_slice($lines, -10);
    echo "<pre>" . implode('', $last_lines) . "</pre>";
} else {
    echo "<p>Error log not accessible from this context. Check your server's error.log.</p>";
}
echo "</div>";

// 8. Troubleshooting Tips
echo "<div class='section'>";
echo "<h2>8. Common Issues & Solutions</h2>";
echo "<ol>";
echo "<li><strong>No salary structure set:</strong> Go to Payroll → Salary Structures and set up Basic Salary for each employee</li>";
echo "<li><strong>Tables missing:</strong> Deactivate and reactivate the plugin to recreate tables</li>";
echo "<li><strong>AJAX not working:</strong> Check browser console (F12) for JavaScript errors</li>";
echo "<li><strong>Nonce errors:</strong> Try logging out and back in, or clear browser cache</li>";
echo "</ol>";
echo "</div>";

echo "<p><a href='" . admin_url('admin.php?page=hrm-payroll') . "'>← Back to Payroll Page</a></p>";
